<?php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/helpers.php';

function pricing_find_rate(int $device_count): array {
  $slabs = db_all("SELECT * FROM pricing_slabs WHERE is_active=1 ORDER BY min_devices ASC");
  foreach ($slabs as $s) {
    if ($device_count >= (int)$s['min_devices'] && $device_count <= (int)$s['max_devices']) {
      return ['slab_id'=>(int)$s['id'],'rate'=>(int)$s['monthly_rate']];
    }
  }
  // Fallback: if above max, choose highest slab rate (or set custom later)
  if ($slabs) {
    $last = end($slabs);
    return ['slab_id'=>(int)$last['id'],'rate'=>(int)$last['monthly_rate']];
  }
  return ['slab_id'=>null,'rate'=>0];
}

function tenant_device_count(int $tenant_id): int {
  $row = db_one("SELECT COUNT(*) c FROM devices WHERE tenant_id=? AND enabled=1 AND deleted_at IS NULL", [$tenant_id]);
  return (int)($row['c'] ?? 0);
}

function billing_snapshot_exists(int $tenant_id, string $period_ym): bool {
  $row = db_one("SELECT id FROM billing_snapshots WHERE tenant_id=? AND snapshot_for_period_ym=?", [$tenant_id,$period_ym]);
  return (bool)$row;
}

function create_billing_snapshot(int $tenant_id, string $period_ym): void {
  $cnt = tenant_device_count($tenant_id);
  $rateInfo = pricing_find_rate($cnt);
  db_exec("INSERT INTO billing_snapshots (tenant_id,snapshot_for_period_ym,snapshot_at,device_count,slab_id,rate_applied,note_json)
           VALUES (?,?,?,?,?,?,?)",
    [$tenant_id,$period_ym, now_dt(), $cnt, $rateInfo['slab_id'], $rateInfo['rate'], json_encode(['rule'=>'snapshot_25th'], JSON_UNESCAPED_UNICODE)]
  );
}

function invoice_exists(int $tenant_id, string $period_ym): bool {
  $row = db_one("SELECT id FROM invoices WHERE tenant_id=? AND period_ym=?", [$tenant_id,$period_ym]);
  return (bool)$row;
}

function create_invoice_from_snapshot(int $tenant_id, string $period_ym, ?int $created_by_user_id=null): ?int {
  $snap = db_one("SELECT * FROM billing_snapshots WHERE tenant_id=? AND snapshot_for_period_ym=?", [$tenant_id,$period_ym]);
  if (!$snap) return null;

  $cfg = app_config();
  $due_days = (int)($cfg['billing']['due_days'] ?? 10);

  $invoice_date = now_date();
  $due_date = (new DateTime($invoice_date))->modify("+{$due_days} days")->format('Y-m-d');

  db_exec("INSERT INTO invoices (tenant_id,period_ym,invoice_date,due_date,status,total_amount,created_by_user_id)
           VALUES (?,?,?,?, 'UNPAID', 0, ?)",
    [$tenant_id,$period_ym,$invoice_date,$due_date,$created_by_user_id]
  );
  $invId = (int)db()->lastInsertId();

  $desc = "Devices ({$snap['device_count']}) monthly billing";
  $qty = 1;
  $unit = (int)$snap['rate_applied'];
  $line = $unit * $qty;

  db_exec("INSERT INTO invoice_items (invoice_id,description,qty,unit_price,line_total) VALUES (?,?,?,?,?)",
    [$invId,$desc,$qty,$unit,$line]
  );

  db_exec("UPDATE invoices SET total_amount=? WHERE id=?", [$line,$invId]);
  return $invId;
}

function invoice_recalculate(int $invoice_id): void {
  $row = db_one("SELECT SUM(line_total) s FROM invoice_items WHERE invoice_id=?", [$invoice_id]);
  $sum = (int)($row['s'] ?? 0);
  db_exec("UPDATE invoices SET total_amount=? WHERE id=?", [$sum,$invoice_id]);
}
