<?php
require_once __DIR__ . '/../core/auth.php';
require_once __DIR__ . '/../core/helpers.php';
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/audit.php';

function staff_page(): void {
  $u = require_login();
  if (!is_tenant_admin($u) && !is_super($u)) { http_response_code(403); exit('Forbidden'); }

  $tenantId = is_super($u) ? (int)($_GET['tenant_id'] ?? 0) : (int)$u['tenant_id'];
  if (!$tenantId) { http_response_code(400); exit('tenant_id required'); }

  $tenant = db_one("SELECT * FROM tenants WHERE id=?", [$tenantId]);
  if (!$tenant) { http_response_code(404); exit('Tenant not found'); }

  $staff = db_all("SELECT id, username, role, status, created_at FROM users WHERE tenant_id=? AND role IN ('tenant_admin','tenant_staff') ORDER BY id DESC", [$tenantId]);

  include __DIR__ . '/../views/staff/index.php';
}

function staff_create_action(): void {
  $u = require_login();
  if (!is_tenant_admin($u) && !is_super($u)) { http_response_code(403); exit('Forbidden'); }

  $tenantId = is_super($u) ? (int)($_POST['tenant_id'] ?? 0) : (int)$u['tenant_id'];
  if (!$tenantId) { http_response_code(400); exit('tenant_id required'); }

  $username = trim($_POST['username'] ?? '');
  $password = (string)($_POST['password'] ?? '');
  $role = $_POST['role'] ?? 'tenant_staff';
  if (!in_array($role, ['tenant_staff'], true)) $role = 'tenant_staff';

  if (!$username || !$password) {
    redirect('index.php?page=staff&tenant_id='.$tenantId.'&err=required');
  }

  $hash = password_hash($password, PASSWORD_BCRYPT);
  try {
    db_exec("INSERT INTO users (tenant_id,username,password_hash,role,status) VALUES (?,?,?,?, 'ACTIVE')",
      [$tenantId, $username, $hash, $role]
    );
  } catch (Exception $e) {
    redirect('index.php?page=staff&tenant_id='.$tenantId.'&err=exists');
  }

  audit_log($tenantId, (int)$u['id'], 'STAFF_CREATED', 'user', (int)db()->lastInsertId(), ['username'=>$username,'role'=>$role]);
  redirect('index.php?page=staff&tenant_id='.$tenantId.'&msg=created');
}

function staff_toggle_action(): void {
  $u = require_login();
  if (!is_tenant_admin($u) && !is_super($u)) { http_response_code(403); exit('Forbidden'); }

  $tenantId = is_super($u) ? (int)($_POST['tenant_id'] ?? 0) : (int)$u['tenant_id'];
  if (!$tenantId) { http_response_code(400); exit('tenant_id required'); }

  $userId = (int)($_POST['user_id'] ?? 0);
  $target = db_one("SELECT * FROM users WHERE id=? AND tenant_id=?", [$userId,$tenantId]);
  if (!$target) { http_response_code(404); exit('User not found'); }

  // hard rule: only super can disable tenant_admin; tenant_admin can only manage tenant_staff
  if (!is_super($u) && $target['role'] === 'tenant_admin') {
    http_response_code(403); exit('Forbidden');
  }

  $newStatus = ($target['status'] === 'ACTIVE') ? 'DISABLED' : 'ACTIVE';
  db_exec("UPDATE users SET status=? WHERE id=?", [$newStatus,$userId]);

  audit_log($tenantId, (int)$u['id'], 'STAFF_STATUS_TOGGLED', 'user', $userId, ['from'=>$target['status'],'to'=>$newStatus]);
  redirect('index.php?page=staff&tenant_id='.$tenantId.'&msg=updated');
}

function staff_reset_password_action(): void {
  $u = require_login();
  if (!is_tenant_admin($u) && !is_super($u)) { http_response_code(403); exit('Forbidden'); }

  $tenantId = is_super($u) ? (int)($_POST['tenant_id'] ?? 0) : (int)$u['tenant_id'];
  if (!$tenantId) { http_response_code(400); exit('tenant_id required'); }

  $userId = (int)($_POST['user_id'] ?? 0);
  $newPass = (string)($_POST['new_password'] ?? '');

  if (!$newPass) {
    redirect('index.php?page=staff&tenant_id='.$tenantId.'&err=pass_required');
  }

  $target = db_one("SELECT * FROM users WHERE id=? AND tenant_id=?", [$userId,$tenantId]);
  if (!$target) { http_response_code(404); exit('User not found'); }

  // hard rule: only super can reset tenant_admin password; tenant_admin can reset tenant_staff
  if (!is_super($u) && $target['role'] === 'tenant_admin') {
    http_response_code(403); exit('Forbidden');
  }

  $hash = password_hash($newPass, PASSWORD_BCRYPT);
  db_exec("UPDATE users SET password_hash=? WHERE id=?", [$hash,$userId]);

  audit_log($tenantId, (int)$u['id'], 'STAFF_PASSWORD_RESET', 'user', $userId, []);
  redirect('index.php?page=staff&tenant_id='.$tenantId.'&msg=updated');
}
