<?php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/crypto.php';
require_once __DIR__ . '/drivers.php';
require_once __DIR__ . '/connectors/ssh.php';
require_once __DIR__ . '/connectors/telnet.php';

function build_backup_dir(array $device): string {
  $cfg = app_config();
  $root = rtrim($cfg['app']['backup_root'], '/');
  $tenantId = (int)$device['tenant_id'];
  $cat = $device['device_category'];
  $brand = $device['brand'];

  $dir = "{$root}/{$tenantId}/{$cat}/{$brand}";
  if (!is_dir($dir)) mkdir($dir, 0770, true);
  return $dir;
}

function build_backup_path(array $device, string $ext): string {
  $dir = build_backup_dir($device);
  $ts = date('Y-m-d_His');
  $safeName = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $device['device_name']);
  $safeIp = preg_replace('/[^0-9a-fA-F\.\:]/', '_', $device['ip_address']);
  return "{$dir}/{$safeName}__{$safeIp}__{$ts}.{$ext}";
}

function resolve_protocol(array $device): string {
  if ($device['protocol'] !== 'default') return $device['protocol'];
  $cfg = app_config();
  return $cfg['defaults']['default_protocol'] ?? 'ssh';
}

function run_device_backup(array $device): array {
  // Returns: ['ok'=>bool,'content'=>string,'ext'=>string,'kind'=>'text|binary','error'=>string]
  $profile = driver_load($device['driver_key']);

  $proto = resolve_protocol($device);
  $host = $device['ip_address'];
  $port = (int)$device['port'];
  $username = $device['username'];
  $password = decrypt_text($device['secret_enc']);
  $commands = $profile['commands'] ?? [];
  $ext = $profile['ext'] ?? 'txt';
  $kind = $profile['backup_kind'] ?? 'text';

  if ($proto === 'ssh') {
    $res = ssh_run_commands($host, $port, $username, $password, $commands);
  } else {
    $res = telnet_run_commands($host, $port, $username, $password, $commands);
  }
  if (!$res['ok']) {
    return ['ok'=>false,'content'=>'','ext'=>$ext,'kind'=>$kind,'error'=>$res['error'] ?: 'Connection failed'];
  }
  return ['ok'=>true,'content'=>$res['output'],'ext'=>$ext,'kind'=>$kind,'error'=>''];
}
