<?php
require_once __DIR__ . '/../app/core/db.php';
require_once __DIR__ . '/../app/core/backup_engine.php';
require_once __DIR__ . '/../app/core/notifications.php';
require_once __DIR__ . '/../app/core/telegram.php';
require_once __DIR__ . '/../app/core/audit.php';

function tenant_is_active(int $tenant_id): bool {
  $t = db_one("SELECT status FROM tenants WHERE id=?", [$tenant_id]);
  return $t && $t['status'] === 'ACTIVE';
}

$job = db_one("SELECT * FROM backups WHERE status='pending' AND scheduled_at <= NOW() ORDER BY scheduled_at ASC LIMIT 1");
if (!$job) exit;

$backupId = (int)$job['id'];
$tenantId = (int)$job['tenant_id'];
$deviceId = (int)$job['device_id'];

if (!tenant_is_active($tenantId)) {
  db_exec("UPDATE backups SET status='failed', error_text=?, finished_at=NOW() WHERE id=?",
    ['Tenant inactive (suspended/disabled); blocked', $backupId]
  );
  exit;
}

db_exec("UPDATE backups SET status='running', started_at=NOW() WHERE id=?", [$backupId]);

$device = db_one("SELECT * FROM devices WHERE id=? AND tenant_id=? AND deleted_at IS NULL", [$deviceId,$tenantId]);
if (!$device) {
  db_exec("UPDATE backups SET status='failed', error_text=?, finished_at=NOW() WHERE id=?", ['Device not found', $backupId]);
  exit;
}

$start = microtime(true);
$res = run_device_backup($device);
$dur = (int)round(microtime(true) - $start);

if (!$res['ok']) {
  db_exec("UPDATE backups SET status='failed', error_text=?, finished_at=NOW(), duration_sec=? WHERE id=?",
    [$res['error'] ?: 'Backup failed', $dur, $backupId]
  );
  notify_create($tenantId, null, 'BACKUP_FAILED', 'Device backup failed',
    "Device: {$device['device_name']} ({$device['ip_address']})\nReason: {$res['error']}", null
  );
  telegram_send($tenantId, "❌ Backup FAILED\nDevice: {$device['device_name']} ({$device['ip_address']})\nReason: {$res['error']}");
  telegram_send(null, "❌ Backup FAILED (Tenant {$tenantId})\nDevice: {$device['device_name']} ({$device['ip_address']})");
  audit_log($tenantId, null, 'BACKUP_FAILED', 'device', (int)$deviceId, ['backup_id'=>$backupId,'reason'=>$res['error']]);
  exit;
}

$path = build_backup_path($device, $res['ext']);
file_put_contents($path, $res['content']);
$size = filesize($path);
$hash = hash_file('sha256', $path);

db_exec("UPDATE backups SET status='success', file_path=?, file_size=?, sha256=?, finished_at=NOW(), duration_sec=? WHERE id=?",
  [$path,$size,$hash,$dur,$backupId]
);
db_exec("UPDATE devices SET last_backup_at=NOW(), next_due_at=DATE_ADD(NOW(), INTERVAL interval_days DAY), updated_at=NOW() WHERE id=?", [$deviceId]);

notify_create($tenantId, null, 'BACKUP_SUCCESS', 'Device backup successfully',
  "Device: {$device['device_name']} ({$device['ip_address']})\nSize: {$size} bytes", null
);
telegram_send($tenantId, "✅ Backup SUCCESS\nDevice: {$device['device_name']} ({$device['ip_address']})\nSize: {$size} bytes");
telegram_send(null, "✅ Backup SUCCESS (Tenant {$tenantId})\nDevice: {$device['device_name']} ({$device['ip_address']})");
audit_log($tenantId, null, 'BACKUP_SUCCESS', 'device', (int)$deviceId, ['backup_id'=>$backupId,'size'=>$size,'sha256'=>$hash]);
