<?php
/**
 * Driver profiles are stored in /app/drivers/{category}/{brand}/{driver}.php
 * Each driver file returns array:
 * [
 *  'title' => '...',
 *  'supports_preview' => true/false,
 *  'ext' => 'txt'|'cfg'|'rsc'|'conf'|'backup',
 *  'backup_kind' => 'text'|'binary',
 *  'commands' => [...],
 * ]
 */
function drivers_root(): string {
  return __DIR__ . '/../drivers';
}

function driver_key_from_path(string $category, string $brand, string $file): string {
  $name = basename($file, '.php');
  return "{$category}.{$brand}.{$name}";
}

function drivers_list(?string $category=null, ?string $brand=null): array {
  $root = drivers_root();
  $out = [];
  $cats = $category ? [$category] : array_filter(scandir($root), fn($x)=>$x[0]!=='.');
  foreach ($cats as $cat) {
    $catDir = $root . '/' . $cat;
    if (!is_dir($catDir)) continue;
    $brands = $brand ? [$brand] : array_filter(scandir($catDir), fn($x)=>$x[0]!=='.');
    foreach ($brands as $br) {
      $brDir = $catDir . '/' . $br;
      if (!is_dir($brDir)) continue;
      foreach (glob($brDir . '/*.php') as $file) {
        $key = driver_key_from_path($cat, $br, $file);
        $profile = include $file;
        $out[] = ['driver_key'=>$key,'category'=>$cat,'brand'=>$br,'profile'=>$profile];
      }
    }
  }
  return $out;
}

function driver_load(string $driver_key): array {
  $parts = explode('.', $driver_key, 3);
  if (count($parts) !== 3) return ['title'=>$driver_key,'supports_preview'=>true,'ext'=>'txt','backup_kind'=>'text','commands'=>[]];
  [$cat,$brand,$name] = $parts;
  $path = drivers_root() . "/{$cat}/{$brand}/{$name}.php";
  if (!file_exists($path)) {
    return ['title'=>$driver_key,'supports_preview'=>true,'ext'=>'txt','backup_kind'=>'text','commands'=>[]];
  }
  $profile = include $path;
  return $profile;
}
