<?php
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/auth.php';
require_once __DIR__ . '/../core/guard.php';
require_once __DIR__ . '/../core/helpers.php';
require_once __DIR__ . '/../core/crypto.php';
require_once __DIR__ . '/../core/audit.php';

function devices_page(): void {
  $u = require_login();

  // hard rule: staff can never delete devices
  if (is_tenant_staff($u)) { http_response_code(403); exit('Forbidden'); }

  require_tenant_active_or_invoice($u, false);

  if (is_super($u)) {
    $tenantId = isset($_GET['tenant_id']) ? (int)$_GET['tenant_id'] : null;
    if ($tenantId) {
      $devices = db_all("SELECT * FROM devices WHERE tenant_id=? AND deleted_at IS NULL ORDER BY id DESC", [$tenantId]);
    } else {
      $devices = db_all("SELECT d.*, t.name tenant_name FROM devices d JOIN tenants t ON t.id=d.tenant_id WHERE d.deleted_at IS NULL ORDER BY d.id DESC LIMIT 200");
    }
  } else {
    $tenantId = tenant_id_of($u);
    $devices = db_all("SELECT * FROM devices WHERE tenant_id=? AND deleted_at IS NULL ORDER BY id DESC", [$tenantId]);
  }

  include __DIR__ . '/../views/devices/index.php';
}

function device_create_action(): void {
  $u = require_login();

  // hard rule: staff can never delete devices
  if (is_tenant_staff($u)) { http_response_code(403); exit('Forbidden'); }

  require_tenant_active_or_invoice($u, false);

  $tenantId = is_super($u) ? (int)($_POST['tenant_id'] ?? 0) : (int)$u['tenant_id'];
  if (!$tenantId) { redirect('index.php?page=devices&err=tenant_required'); }
  $t = tenant_row($tenantId);
  if (!$t) { http_response_code(400); exit('Tenant invalid'); }

  $device_name = trim($_POST['device_name'] ?? '');
  $ip = trim($_POST['ip_address'] ?? '');
  $cat = $_POST['device_category'] ?? 'router';
  $brand = $_POST['brand'] ?? '';
  $driver_key = $_POST['driver_key'] ?? '';
  $protocol = $_POST['protocol'] ?? 'default';
  $port = (int)($_POST['port'] ?? 22);
  $username = trim($_POST['username'] ?? '');
  $password = (string)($_POST['password'] ?? '');
  $interval_days = (int)($_POST['interval_days'] ?? 7);

  if (!$device_name || !$ip || !$brand || !$driver_key || !$username) {
    http_response_code(400); exit('Missing required fields');
  }

  $exists = db_one("SELECT id FROM devices WHERE tenant_id=? AND ip_address=? AND device_category=? AND deleted_at IS NULL", [$tenantId,$ip,$cat]);
  if ($exists) { http_response_code(409); exit('Device already exists'); }

  $secret_enc = encrypt_text($password);

  db_exec("INSERT INTO devices (tenant_id,device_name,ip_address,device_category,brand,driver_key,protocol,port,username,secret_enc,interval_days,enabled,next_due_at)
           VALUES (?,?,?,?,?,?,?,?,?,?,?,1, NOW())",
    [$tenantId,$device_name,$ip,$cat,$brand,$driver_key,$protocol,$port,$username,$secret_enc,$interval_days]
  );
  $deviceId = (int)db()->lastInsertId();

  // enqueue first backup
  db_exec("INSERT INTO backups (tenant_id,device_id,trigger_source,status,scheduled_at) VALUES (?,?,?,?,NOW())",
    [$tenantId,$deviceId,'on_device_add','pending']
  );

  audit_log($tenantId, (int)$u['id'], 'DEVICE_CREATED', 'device', $deviceId, ['auto_first_backup'=>true]);

  redirect('index.php?page=devices&msg=device_added_backup_queued');
}

function device_delete_action(): void {
  $u = require_login();

  // hard rule: staff can never delete devices
  if (is_tenant_staff($u)) { http_response_code(403); exit('Forbidden'); }

  require_tenant_active_or_invoice($u, false);

  $deviceId = (int)($_POST['device_id'] ?? 0);
  $device = db_one("SELECT * FROM devices WHERE id=? AND deleted_at IS NULL", [$deviceId]);
  if (!$device) { http_response_code(404); exit('Not found'); }

  $tenant = tenant_row((int)$device['tenant_id']);
  if (!$tenant) { http_response_code(404); exit('Tenant not found'); }

  // Tenant isolation
  if (!is_super($u) && (int)$device['tenant_id'] !== (int)$u['tenant_id']) {
    http_response_code(403); exit('Forbidden');
  }

  if (!can_device_delete($u, $tenant)) {
    http_response_code(403); exit('Device delete not allowed');
  }

  // Soft delete
  db_exec("UPDATE devices SET deleted_at=NOW() WHERE id=?", [$deviceId]);
  audit_log((int)$device['tenant_id'], (int)$u['id'], 'DEVICE_DELETED', 'device', $deviceId, ['soft_delete'=>true]);

  // Optional immediate purge (or do via cron)
  if ((int)$tenant['purge_files_on_device_delete'] === 1) {
    require_once __DIR__ . '/../core/storage.php';
    purge_device_files((int)$device['tenant_id'], $device);
  }

  redirect('index.php?page=devices&msg=device_deleted');
}
