<?php
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/auth.php';
require_once __DIR__ . '/../core/guard.php';
require_once __DIR__ . '/../core/helpers.php';
require_once __DIR__ . '/../core/drivers.php';

function backups_browse_page(): void {
  $u = require_login();
  require_tenant_active_or_invoice($u, false);

  $cfg = app_config();
  $root = rtrim($cfg['app']['backup_root'], '/');

  $tenantId = is_super($u) ? (int)($_GET['tenant_id'] ?? 0) : (int)$u['tenant_id'];
  if (!$tenantId) { http_response_code(400); exit('Tenant required'); }

  if (!is_super($u) && $tenantId !== (int)$u['tenant_id']) { http_response_code(403); exit('Forbidden'); }

  $dir = $_GET['dir'] ?? '';
  $dir = trim($dir, '/');

  $base = "{$root}/{$tenantId}";
  $current = $dir ? ($base . '/' . $dir) : $base;
  $realBase = realpath($base);
  if (!is_dir($current)) @mkdir($current, 0770, true);
  $realCurrent = realpath($current);

  if (!$realBase || !$realCurrent || strpos($realCurrent, $realBase) !== 0) {
    http_response_code(400); exit('Invalid dir');
  }

  $items = [];
  foreach (scandir($realCurrent) as $x) {
    if ($x === '.' || $x === '..') continue;
    $path = $realCurrent . '/' . $x;
    $items[] = [
      'name' => $x,
      'is_dir' => is_dir($path),
      'size' => is_file($path) ? filesize($path) : null,
      'mtime' => filemtime($path),
    ];
  }

  include __DIR__ . '/../views/backups/browse.php';
}

function backup_preview_page(): void {
  $u = require_login();
  require_tenant_active_or_invoice($u, false);

  $backupId = (int)($_GET['id'] ?? 0);
  $b = db_one("SELECT b.*, d.driver_key FROM backups b JOIN devices d ON d.id=b.device_id WHERE b.id=?", [$backupId]);
  if (!$b) { http_response_code(404); exit('Not found'); }

  if (!is_super($u) && (int)$b['tenant_id'] !== (int)$u['tenant_id']) { http_response_code(403); exit('Forbidden'); }

  $profile = driver_load($b['driver_key']);
  if (empty($profile['supports_preview']) || ($profile['backup_kind'] ?? 'text') !== 'text') {
    http_response_code(403); exit('Preview not allowed');
  }

  $path = $b['file_path'];
  if (!$path || !file_exists($path)) { http_response_code(404); exit('File missing'); }

  $content = file_get_contents($path, false, null, 0, 1024*1024); // 1MB preview
  include __DIR__ . '/../views/backups/preview.php';
}

function backup_download_action(): void {
  $u = require_login();
  require_tenant_active_or_invoice($u, false);

  $backupId = (int)($_GET['id'] ?? 0);
  $b = db_one("SELECT * FROM backups WHERE id=?", [$backupId]);
  if (!$b) { http_response_code(404); exit('Not found'); }

  if (!is_super($u) && (int)$b['tenant_id'] !== (int)$u['tenant_id']) { http_response_code(403); exit('Forbidden'); }

  $path = $b['file_path'];
  if (!$path || !file_exists($path)) { http_response_code(404); exit('File missing'); }

  header('Content-Type: application/octet-stream');
  header('Content-Disposition: attachment; filename="'.basename($path).'"');
  readfile($path);
  exit;
}
