<?php
// cron/daily.php
// Safe to run multiple times a day (idempotent checks).

require_once __DIR__ . '/../app/core/db.php';
require_once __DIR__ . '/../app/core/helpers.php';
require_once __DIR__ . '/../app/core/audit.php';

date_default_timezone_set('Asia/Dhaka');

function ym(DateTime $dt): string { return $dt->format('Y-m'); }

function month_add(DateTime $dt, int $months): DateTime {
  $copy = clone $dt;
  $copy->modify(($months>=0?'+':'').$months.' month');
  return $copy;
}

function get_price_for_device_count(int $count): float {
  $rows = db_all("SELECT min_devices,max_devices,monthly_rate FROM pricing_slabs ORDER BY min_devices ASC");
  foreach ($rows as $r) {
    if ($count >= (int)$r['min_devices'] && $count <= (int)$r['max_devices']) return (float)$r['monthly_rate'];
  }
  // fallback: highest slab
  if ($rows) return (float)$rows[count($rows)-1]['monthly_rate'];
  return 0.0;
}

function notify_tenant(int $tenantId, string $type, string $title, string $body): void {
  db_exec("INSERT INTO notifications (tenant_id,type,title,body,is_read,created_at) VALUES (?,?,?,?,0,NOW())",
    [$tenantId,$type,$title,$body]
  );
}

function cron_snapshot_25th(DateTime $now): void {
  // On 25th, snapshot device count for NEXT month (period_ym)
  if ((int)$now->format('j') !== 25) return;

  $next = month_add($now, 1);
  $period = ym($next);

  $tenants = db_all("SELECT id,name,status FROM tenants WHERE status IN ('ACTIVE','SUSPENDED_UNPAID')");

  foreach ($tenants as $t) {
    $tid = (int)$t['id'];

    $exists = db_one("SELECT id FROM billing_snapshots WHERE tenant_id=? AND period_ym=? LIMIT 1", [$tid,$period]);
    if ($exists) continue;

    $count = db_one("SELECT COUNT(*) AS c FROM devices WHERE tenant_id=? AND deleted_at IS NULL", [$tid]);
    $deviceCount = (int)($count['c'] ?? 0);

    db_exec("INSERT INTO billing_snapshots (tenant_id, period_ym, device_count, created_at) VALUES (?,?,?,NOW())",
      [$tid,$period,$deviceCount]
    );

    audit_log($tid, null, 'BILLING_SNAPSHOT_CREATED', 'billing_snapshot', null, ['period'=>$period,'device_count'=>$deviceCount]);

    notify_tenant($tid, 'BILLING_SNAPSHOT', 'Device count snapshot created',
      "Period: {$period}\nDevice count: {$deviceCount}\nInvoice will be generated on 1st day.");
  }
}

function cron_generate_invoices_1st(DateTime $now): void {
  if ((int)$now->format('j') !== 1) return;

  $period = ym($now);
  $invoiceDate = $now->format('Y-m-01');
  $dueDate = (clone $now)->modify('+10 day')->format('Y-m-d');

  $tenants = db_all("SELECT id,name,status FROM tenants WHERE status IN ('ACTIVE','SUSPENDED_UNPAID')");

  foreach ($tenants as $t) {
    $tid = (int)$t['id'];

    $exists = db_one("SELECT id FROM invoices WHERE tenant_id=? AND period_ym=? LIMIT 1", [$tid,$period]);
    if ($exists) continue;

    // Prefer snapshot made on 25th for this period
    $snap = db_one("SELECT device_count FROM billing_snapshots WHERE tenant_id=? AND period_ym=? LIMIT 1", [$tid,$period]);
    if ($snap) $deviceCount = (int)$snap['device_count'];
    else {
      $count = db_one("SELECT COUNT(*) AS c FROM devices WHERE tenant_id=? AND deleted_at IS NULL", [$tid]);
      $deviceCount = (int)($count['c'] ?? 0);
    }

    $amount = get_price_for_device_count($deviceCount);

    db_exec("INSERT INTO invoices (tenant_id, period_ym, invoice_date, due_date, status, total_amount, created_at) VALUES (?,?,?,?, 'UNPAID', ?, NOW())",
      [$tid,$period,$invoiceDate,$dueDate,$amount]
    );
    $invId = (int)db()->lastInsertId();

    db_exec("INSERT INTO invoice_items (invoice_id, description, amount) VALUES (?,?,?)",
      [$invId, "Monthly service fee ({$deviceCount} devices)", $amount]
    );

    audit_log($tid, null, 'INVOICE_CREATED', 'invoice', $invId, ['period'=>$period,'device_count'=>$deviceCount,'amount'=>$amount]);

    notify_tenant($tid, 'INVOICE_CREATED', 'Invoice created',
      "Invoice #{$invId}\nPeriod: {$period}\nDevices: {$deviceCount}\nAmount: {$amount}\nDue: {$dueDate}");
  }
}

function cron_reminder_before_due(DateTime $now): void {
  // Send reminder 3 days before due for unpaid invoices (once)
  $target = (clone $now)->modify('+3 day')->format('Y-m-d');
  $rows = db_all("SELECT id, tenant_id, due_date, total_amount, period_ym FROM invoices WHERE status='UNPAID' AND due_date=?",
    [$target]
  );

  foreach ($rows as $inv) {
    $invId = (int)$inv['id'];
    $tid = (int)$inv['tenant_id'];

    $already = db_one("SELECT id FROM audit_logs WHERE tenant_id=? AND action_key='BILLING_REMINDER_3D_SENT' AND entity_type='invoice' AND entity_id=? LIMIT 1",
      [$tid,$invId]
    );
    if ($already) continue;

    notify_tenant($tid, 'INVOICE_REMINDER', 'Invoice due soon',
      "Invoice #{$invId} is due on {$inv['due_date']} (in 3 days).\nAmount: {$inv['total_amount']}\nPay to avoid suspension."
    );

    audit_log($tid, null, 'BILLING_REMINDER_3D_SENT', 'invoice', $invId, ['due_date'=>$inv['due_date']]);
  }
}

function cron_suspend_overdue(DateTime $now): void {
  $today = $now->format('Y-m-d');

  $rows = db_all("SELECT i.id AS invoice_id, i.tenant_id, i.due_date, t.status
                  FROM invoices i
                  JOIN tenants t ON t.id=i.tenant_id
                  WHERE i.status='UNPAID' AND i.due_date < ? AND t.status='ACTIVE'", [$today]);

  foreach ($rows as $r) {
    $tid = (int)$r['tenant_id'];
    db_exec("UPDATE tenants SET status='SUSPENDED_UNPAID' WHERE id=?", [$tid]);

    audit_log($tid, null, 'TENANT_SUSPENDED_UNPAID', 'tenant', $tid, ['invoice_id'=>(int)$r['invoice_id'],'due_date'=>$r['due_date']]);

    notify_tenant($tid, 'TENANT_SUSPENDED', 'Billing activities suspended',
      "Due to unpaid invoice #{$r['invoice_id']} (due {$r['due_date']}), your billing activities are suspended.\nPlease pay to re-enable service."
    );
  }
}

$now = new DateTime('now');

cron_snapshot_25th($now);
cron_generate_invoices_1st($now);
cron_reminder_before_due($now);
cron_suspend_overdue($now);

echo "OK ". $now->format('Y-m-d H:i:s') . PHP_EOL;
