<?php
/**
 * bKash PGW helper
 * Modes:
 * - merchant_url: Redirect to external Merchant Checkout URL you provide
 * - tokenized: Use Tokenized Checkout API (skeleton)
 */

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/settings.php';
require_once __DIR__ . '/helpers.php';

function bkash_settings_get(): array {
  $rows = db_all("SELECT k,v FROM settings WHERE scope='GLOBAL' AND k LIKE 'bkash_%'");
  $out = [];
  foreach ($rows as $r) $out[$r['k']] = $r['v'];
  return $out;
}

function bkash_mode(): string {
  $s = bkash_settings_get();
  $m = $s['bkash_mode'] ?? 'merchant_url';
  return in_array($m, ['merchant_url','tokenized'], true) ? $m : 'merchant_url';
}

function bkash_callback_secret(): string {
  $s = bkash_settings_get();
  return (string)($s['bkash_callback_secret'] ?? '');
}

function bkash_sign_callback(int $invoice_id, string $trx_id, string $status, float $amount): string {
  $secret = bkash_callback_secret();
  $data = $invoice_id . '|' . $trx_id . '|' . $status . '|' . number_format($amount, 2, '.', '');
  return hash_hmac('sha256', $data, $secret);
}

function bkash_verify_callback_sig(int $invoice_id, string $trx_id, string $status, float $amount, string $sig): bool {
  if (!$sig) return false;
  $exp = bkash_sign_callback($invoice_id, $trx_id, $status, $amount);
  return hash_equals($exp, $sig);
}

function bkash_create_payment_link(int $invoice_id): string {
  $s = bkash_settings_get();
  $enabled = (($s['bkash_enabled'] ?? '0') === '1');
  if (!$enabled) return "index.php?page=invoice_pay&id=" . $invoice_id;

  $mode = bkash_mode();
  $payUrl = trim((string)($s['bkash_pay_url'] ?? ''));

  if ($mode === 'merchant_url') {
    if (!$payUrl) return "index.php?page=invoice_pay&id=" . $invoice_id;

    $inv = db_one("SELECT id, tenant_id, total_amount FROM invoices WHERE id=?", [$invoice_id]);
    if (!$inv) return "index.php?page=invoices";

    $base = site_url();
    $qs = http_build_query([
      'invoice_id' => (int)$inv['id'],
      'tenant_id'  => (int)$inv['tenant_id'],
      'amount'     => (float)$inv['total_amount'],
      'callback'   => $base . "/index.php?action=bkash_callback",
      'success'    => $base . "/index.php?page=invoices&msg=paid",
      'fail'       => $base . "/index.php?page=invoice_pay&id=" . (int)$inv['id'] . "&err=failed",
    ]);

    $hasQ = (strpos($payUrl, '?') !== false);
    return rtrim($payUrl, '?&') . ($hasQ ? '&' : '?') . $qs;
  }

  // tokenized mode (skeleton) - route to internal handler page (future)
  return "index.php?page=invoice_pay&id=" . $invoice_id . "&mode=tokenized";
}
