<?php
/**
 * bKash Tokenized Checkout (Skeleton)
 * NOTE: This is a wiring/skeleton only. You must configure credentials + endpoints.
 * Supports:
 * - token grant
 * - create payment
 * - execute payment
 */

require_once __DIR__ . '/settings.php';
require_once __DIR__ . '/helpers.php';

function bkash_cfg(string $k, string $default=''): string {
  return (string)setting_get('GLOBAL', null, $k, $default);
}

function bkash_base_url(): string {
  return rtrim(bkash_cfg('bkash_base_url', ''), '/');
}

function bkash_http(string $method, string $url, array $headers=[], ?string $body=null): array {
  $ch = curl_init();
  curl_setopt($ch, CURLOPT_URL, $url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
  curl_setopt($ch, CURLOPT_TIMEOUT, 45);
  if ($body !== null) curl_setopt($ch, CURLOPT_POSTFIELDS, $body);

  $hdr = [];
  foreach ($headers as $k=>$v) $hdr[] = $k.': '.$v;
  if ($hdr) curl_setopt($ch, CURLOPT_HTTPHEADER, $hdr);

  $resp = curl_exec($ch);
  $err  = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  $json = is_string($resp) ? json_decode($resp, true) : null;
  return ['ok'=>($code>=200 && $code<300),'code'=>$code,'error'=>$err,'raw'=>$resp,'json'=>$json];
}

function bkash_get_token(): array {
  $base = bkash_base_url();
  $url = $base . '/tokenized/checkout/token/grant';
  $body = json_encode([
    'app_key' => bkash_cfg('bkash_app_key',''),
    'app_secret' => bkash_cfg('bkash_app_secret',''),
  ]);

  $headers = [
    'Content-Type'=>'application/json',
    'Accept'=>'application/json',
    'username'=>bkash_cfg('bkash_username',''),
    'password'=>bkash_cfg('bkash_password',''),
  ];

  return bkash_http('POST', $url, $headers, $body);
}

function bkash_create_payment(string $token, array $payload): array {
  $base = bkash_base_url();
  $url = $base . '/tokenized/checkout/create';
  $headers = [
    'Content-Type'=>'application/json',
    'Accept'=>'application/json',
    'authorization'=>$token,
    'x-app-key'=>bkash_cfg('bkash_app_key',''),
  ];
  return bkash_http('POST', $url, $headers, json_encode($payload));
}

function bkash_execute_payment(string $token, string $paymentID): array {
  $base = bkash_base_url();
  $url = $base . '/tokenized/checkout/execute';
  $headers = [
    'Content-Type'=>'application/json',
    'Accept'=>'application/json',
    'authorization'=>$token,
    'x-app-key'=>bkash_cfg('bkash_app_key',''),
  ];
  $payload = ['paymentID'=>$paymentID];
  return bkash_http('POST', $url, $headers, json_encode($payload));
}
