<?php
require_once __DIR__ . '/../core/db.php';
require_once __DIR__ . '/../core/auth.php';
require_once __DIR__ . '/../core/guard.php';
require_once __DIR__ . '/../core/helpers.php';
require_once __DIR__ . '/../core/billing.php';
require_once __DIR__ . '/../core/pgw_bkash.php';
require_once __DIR__ . '/../core/notifications.php';
require_once __DIR__ . '/../core/telegram.php';
require_once __DIR__ . '/../core/audit.php';

function invoice_lock_paid(int $invoiceId): void {
  $exists = db_one("SELECT id FROM audit_logs WHERE action_key='INVOICE_PAID_LOCK' AND entity_type='invoice' AND entity_id=? LIMIT 1", [$invoiceId]);
  if (!$exists) {
    audit_log(null, null, 'INVOICE_PAID_LOCK', 'invoice', $invoiceId, []);
  }
}

function invoice_is_locked(int $invoiceId): bool {
  $row = db_one("SELECT id FROM audit_logs WHERE action_key='INVOICE_PAID_LOCK' AND entity_type='invoice' AND entity_id=? LIMIT 1", [$invoiceId]);
  return (bool)$row;
}

function bkash_callback_action(): void {
  $invoiceId = (int)($_REQUEST['invoice_id'] ?? 0);
  $trxId = trim((string)($_REQUEST['trx_id'] ?? ''));
  $status = trim((string)($_REQUEST['status'] ?? ''));
  $amount = (float)($_REQUEST['amount'] ?? 0);
  $sig = trim((string)($_REQUEST['sig'] ?? ''));

  if (!$invoiceId || !$trxId || !$status || !$amount) {
    http_response_code(400); echo "bad_request"; return;
  }

  if (!bkash_verify_callback_sig($invoiceId, $trxId, $status, $amount, $sig)) {
    http_response_code(401); echo "invalid_signature"; return;
  }

  $inv = db_one("SELECT * FROM invoices WHERE id=?", [$invoiceId]);
  if (!$inv) { http_response_code(404); echo "invoice_not_found"; return; }

  if (abs(((float)$inv['total_amount']) - $amount) > 0.01) {
    http_response_code(400); echo "amount_mismatch"; return;
  }

  $tenantId = (int)$inv['tenant_id'];

  if (strtolower($status) !== 'success') {
    audit_log($tenantId, null, 'BKASH_CALLBACK_FAILED', 'invoice', $invoiceId, ['trx_id'=>$trxId,'status'=>$status]);
    echo "ignored"; return;
  }

  if ($inv['status'] !== 'PAID') {
    db_exec("UPDATE invoices SET status='PAID', last_paid_at=NOW() WHERE id=?", [$invoiceId]);
    invoice_lock_paid($invoiceId);
  }

  $pay = db_one("SELECT id FROM payments WHERE invoice_id=? AND trx_id=? LIMIT 1", [$invoiceId,$trxId]);
  if (!$pay) {
    db_exec("INSERT INTO payments (invoice_id, provider, trx_id, amount, status, paid_at) VALUES (?,?,?,?, 'SUCCESS', NOW())",
      [$invoiceId, 'bkash', $trxId, $amount]
    );
  }

  $t = db_one("SELECT * FROM tenants WHERE id=?", [$tenantId]);
  if ($t && $t['status'] === 'SUSPENDED_UNPAID') {
    $over = db_one("SELECT COUNT(*) c FROM invoices WHERE tenant_id=? AND status='UNPAID' AND due_date < CURDATE()", [$tenantId]);
    if ((int)($over['c'] ?? 0) === 0) {
      db_exec("UPDATE tenants SET status='ACTIVE' WHERE id=?", [$tenantId]);
      audit_log($tenantId, null, 'TENANT_UNSUSPENDED_AFTER_PAYMENT', 'tenant', $tenantId, ['invoice_id'=>$invoiceId,'trx_id'=>$trxId]);
    }
  }

  audit_log($tenantId, null, 'BKASH_CALLBACK_SUCCESS', 'invoice', $invoiceId, ['trx_id'=>$trxId,'amount'=>$amount]);
  echo "ok";
}


function invoices_page(): void {
  $u = require_login();
  // allow invoice pages even if suspended
  require_tenant_active_or_invoice($u, true);

  if (is_super($u)) {
    $tenantId = (int)($_GET['tenant_id'] ?? 0);
    $invoices = $tenantId ? db_all("SELECT * FROM invoices WHERE tenant_id=? ORDER BY id DESC", [$tenantId])
                         : db_all("SELECT i.*, t.name tenant_name FROM invoices i JOIN tenants t ON t.id=i.tenant_id ORDER BY i.id DESC LIMIT 200");
  } else {
    $tenantId = (int)$u['tenant_id'];
    $invoices = db_all("SELECT * FROM invoices WHERE tenant_id=? ORDER BY id DESC", [$tenantId]);
  }

  include __DIR__ . '/../views/billing/invoices.php';
}

function invoice_pay_page(): void {
  $u = require_login();
  require_tenant_active_or_invoice($u, true);

  $id = (int)($_GET['id'] ?? 0);
  $inv = db_one("SELECT * FROM invoices WHERE id=?", [$id]);
  if (!$inv) { http_response_code(404); exit('Not found'); }

  if (!is_super($u) && (int)$inv['tenant_id'] !== (int)$u['tenant_id']) { http_response_code(403); exit('Forbidden'); }

  $pay_link = bkash_create_payment_link($id);
  include __DIR__ . '/../views/billing/pay.php';
}

// Simulated payment success endpoint (replace with real PGW callback)
function invoice_pay_success_action(): void {
  $u = require_login();
  require_tenant_active_or_invoice($u, true);

  $id = (int)($_GET['id'] ?? 0);
  $inv = db_one("SELECT * FROM invoices WHERE id=?", [$id]);
  if (!$inv) { http_response_code(404); exit('Not found'); }

  if (!is_super($u) && (int)$inv['tenant_id'] !== (int)$u['tenant_id']) { http_response_code(403); exit('Forbidden'); }

  if ($inv['status'] !== 'PAID') {
    db_exec("UPDATE invoices SET status='PAID', last_paid_at=NOW() WHERE id=?", [$id]);
    // Unsuspend tenant if it was suspended unpaid AND no overdue unpaid invoices remain
    $t = db_one("SELECT * FROM tenants WHERE id=?", [(int)$inv['tenant_id']]);
    if ($t && $t['status'] === 'SUSPENDED_UNPAID') {
      $over = db_one("SELECT COUNT(*) c FROM invoices WHERE tenant_id=? AND status='UNPAID' AND due_date < CURDATE()", [(int)$t['id']]);
      if ((int)($over['c'] ?? 0) === 0) {
        db_exec("UPDATE tenants SET status='ACTIVE', suspended_reason=NULL, suspended_at=NULL, enabled_at=NOW() WHERE id=?", [(int)$t['id']]);
        audit_log((int)$t['id'], (int)$u['id'], 'TENANT_UNSUSPENDED_AFTER_PAYMENT', 'tenant', (int)$t['id'], []);
      }
    }

    notify_create((int)$inv['tenant_id'], null, 'PAYMENT_SUCCESS', 'Payment received', "Invoice #{$id} paid successfully", 'index.php?page=invoices');
    telegram_send((int)$inv['tenant_id'], "✅ Payment SUCCESS\nInvoice #{$id} (Period {$inv['period_ym']}) paid.");
    telegram_send(null, "✅ Tenant payment\nTenant ID {$inv['tenant_id']} Invoice #{$id} PAID.");
    audit_log((int)$inv['tenant_id'], (int)$u['id'], 'PAYMENT_SUCCESS', 'invoice', $id, []);
  }

  redirect('index.php?page=invoices&msg=paid');
}

// Superadmin: add adjustment line (discount/extra)
function invoice_add_adjustment_action(): void {
  $u = require_login();
  if (!is_super($u)) { http_response_code(403); exit('Forbidden'); }

  $invoice_id = (int)($_POST['invoice_id'] ?? 0);
  $desc = trim($_POST['description'] ?? 'Adjustment');
  $amount = (int)($_POST['amount'] ?? 0); // negative=discount, positive=extra

  $inv = db_one("SELECT * FROM invoices WHERE id=?", [$invoice_id]);
  if (!$inv) { http_response_code(404); exit('Not found'); }

  db_exec("INSERT INTO invoice_items (invoice_id,description,qty,unit_price,line_total) VALUES (?,?,?,?,?)",
    [$invoice_id, $desc, 1, $amount, $amount]
  );
  invoice_recalculate($invoice_id);
  audit_log(null, (int)$u['id'], 'INVOICE_ADJUSTMENT_ADDED', 'invoice', $invoice_id, ['desc'=>$desc,'amount'=>$amount]);

  redirect('index.php?page=invoices&tenant_id='.(int)$inv['tenant_id'].'&msg=adjusted');
}
